"""
Results for Kalman Filter

Author: Chad Fulton
License: Simplified-BSD

Notes
-----

There was an error in the loglikelihood computation in the GAUSS code from
Kim and Nelson for the Clark 1989 model, omitting the exponent term on
``2 pi''. Therefore the loglikelihood in the test results for uc_bi is instead
from the FKF R library, which can be replicated using the
``test_clark1989_r.R'' file.

References
----------

Datasets produced using GAUSS code described in Kim and Nelson (1999) and
found at http://econ.korea.ac.kr/~cjkim/SSMARKOV.htm
Accessed 2013-11-18
Code was run using OxGauss 7.

Kim, Chang-Jin, and Charles R. Nelson. 1999.
"State-Space Models with Regime Switching:
Classical and Gibbs-Sampling Approaches with Applications".
MIT Press Books. The MIT Press.

"""

"""
Clark's (1987) Univariate Unobserved Components model of GDP

See http://econ.korea.ac.kr/~cjkim/MARKOV/prgmlist.htm#chapter3

Gauss Code: UC_UNI.OPT
Dataset:    GDP4795.PRN
"""
uc_uni = {
    'data': [
        1239.5, 1247.19995117, 1255, 1269.5, 1284, 1295.69995117,
        1303.80004883, 1316.40002441, 1305.30004883, 1302, 1312.59997559,
        1301.90002441, 1350.90002441, 1393.5, 1445.19995117, 1484.5,
        1504.09997559, 1548.30004883, 1585.40002441, 1596, 1607.69995117,
        1612.09997559, 1621.90002441, 1657.80004883, 1687.30004883,
        1695.30004883, 1687.90002441, 1671.19995117, 1660.80004883,
        1658.40002441, 1677.69995117, 1698.30004883, 1742.5, 1758.59997559,
        1778.19995117, 1793.90002441, 1787, 1798.5, 1802.19995117,
        1826.59997559, 1836.40002441, 1834.80004883, 1851.19995117, 1830.5,
        1790.09997559, 1804.40002441, 1840.90002441, 1880.90002441,
        1904.90002441, 1937.5, 1930.80004883, 1941.90002441, 1976.90002441,
        1971.69995117, 1973.69995117, 1961.09997559, 1977.40002441, 2006,
        2035.19995117, 2076.5, 2103.80004883, 2125.69995117, 2142.60009766,
        2140.19995117, 2170.89990234, 2199.5, 2237.60009766, 2254.5,
        2311.10009766, 2329.89990234, 2357.39990234, 2364, 2410.10009766,
        2442.80004883, 2485.5, 2543.80004883, 2596.80004883, 2601.39990234,
        2626.10009766, 2640.5, 2657.19995117, 2669, 2699.5, 2715.10009766,
        2752.10009766, 2796.89990234, 2816.80004883, 2821.69995117,
        2864.60009766, 2867.80004883, 2884.5, 2875.10009766, 2867.80004883,
        2859.5, 2895, 2873.30004883, 2939.89990234, 2944.19995117,
        2962.30004883, 2977.30004883, 3037.30004883, 3089.69995117,
        3125.80004883, 3175.5, 3253.30004883, 3267.60009766, 3264.30004883,
        3289.10009766, 3259.39990234, 3267.60009766, 3239.10009766,
        3226.39990234, 3154, 3190.39990234, 3249.89990234, 3292.5,
        3356.69995117, 3369.19995117, 3381, 3416.30004883, 3466.39990234, 3525,
        3574.39990234, 3567.19995117, 3591.80004883, 3707, 3735.60009766,
        3779.60009766, 3780.80004883, 3784.30004883, 3807.5, 3814.60009766,
        3830.80004883, 3732.60009766, 3733.5, 3808.5, 3860.5, 3844.39990234,
        3864.5, 3803.10009766, 3756.10009766, 3771.10009766, 3754.39990234,
        3759.60009766, 3783.5, 3886.5, 3944.39990234, 4012.10009766, 4089.5,
        4144, 4166.39990234, 4194.20019531, 4221.79980469, 4254.79980469, 4309,
        4333.5, 4390.5, 4387.70019531, 4412.60009766, 4427.10009766, 4460,
        4515.29980469, 4559.29980469, 4625.5, 4655.29980469, 4704.79980469,
        4734.5, 4779.70019531, 4817.60009766, 4839, 4839, 4856.70019531,
        4898.29980469, 4917.10009766, 4906.5, 4867.20019531, 4842,
        4867.89990234, 4879.89990234, 4880.79980469, 4918.5, 4947.5, 4990.5,
        5060.70019531, 5075.29980469, 5105.39990234, 5139.39990234, 5218,
        5261.10009766, 5314.10009766, 5367, 5433.79980469, 5470.10009766,
        5487.79980469, 5544.60009766
    ],
    'start': 21-1,
    'parameters': [0.005539, 0.006164, 0.000184, 1.531659, -0.585422],
    'standard_errors': [0.000000, 0.000000, 0.000000, 0.000000, 0.000000],
    'loglike': 578.520887,
    'states': [
        [7.369243,  0.013317,  0.018762],
        [7.374220,  0.011073,  0.016581],
        [7.385527,  0.005826,  0.015828],
        [7.416916, -0.003669,  0.017851],
        [7.434130, -0.003245,  0.017775],
        [7.433819,  0.001796,  0.015781],
        [7.427416,  0.003825,  0.013508],
        [7.419109,  0.002189,  0.011420],
        [7.419253, -0.004198,  0.010408],
        [7.424214, -0.010605,  0.009947],
        [7.443116, -0.017937,  0.010663],
        [7.455732, -0.018349,  0.010810],
        [7.483689, -0.020612,  0.012044],
        [7.484287, -0.012014,  0.011259],
        [7.493577, -0.010220,  0.011130],
        [7.500748, -0.008601,  0.010882],
        [7.493715, -0.005422,  0.009805],
        [7.504849, -0.010140,  0.009882],
        [7.507452, -0.010689,  0.009477],
        [7.524476, -0.014264,  0.009882],
        [7.527391, -0.011828,  0.009521],
        [7.525478, -0.010787,  0.008949],
        [7.537758, -0.014169,  0.009110],
        [7.523893, -0.011549,  0.008031],
        [7.504850, -0.014823,  0.006795],
        [7.524920, -0.026937,  0.007384],
        [7.547145, -0.029135,  0.008024],
        [7.563759, -0.024254,  0.008385],
        [7.569019, -0.016834,  0.008257],
        [7.582139, -0.012985,  0.008452],
        [7.573090, -0.007400,  0.007768],
        [7.581280, -0.009858,  0.007784],
        [7.600320, -0.011035,  0.008205],
        [7.593454, -0.006802,  0.007653],
        [7.596716, -0.009050,  0.007495],
        [7.592481, -0.011220,  0.007082],
        [7.605464, -0.015926,  0.007286],
        [7.619716, -0.015818,  0.007523],
        [7.630873, -0.012524,  0.007644],
        [7.647019, -0.008580,  0.007924],
        [7.654471, -0.002971,  0.007909],
        [7.661563,  0.000293,  0.007883],
        [7.667909,  0.001866,  0.007834],
        [7.666825,  0.001830,  0.007557],
        [7.683710, -0.000813,  0.007843],
        [7.695061,  0.000925,  0.007950],
        [7.709743,  0.003416,  0.008151],
        [7.714033,  0.006650,  0.008037],
        [7.737525,  0.007954,  0.008490],
        [7.740667,  0.012914,  0.008335],
        [7.751712,  0.013603,  0.008412],
        [7.754422,  0.013689,  0.008250],
        [7.775189,  0.012235,  0.008603],
        [7.785979,  0.014922,  0.008664],
        [7.801228,  0.017002,  0.008847],
        [7.820961,  0.020453,  0.009147],
        [7.836561,  0.025474,  0.009324],
        [7.835776,  0.028029,  0.009049],
        [7.848074,  0.025181,  0.009137],
        [7.855709,  0.023015,  0.009097],
        [7.864971,  0.020057,  0.009101],
        [7.872361,  0.017098,  0.009056],
        [7.886320,  0.014502,  0.009185],
        [7.893144,  0.013440,  0.009123],
        [7.908033,  0.012086,  0.009274],
        [7.923091,  0.013175,  0.009425],
        [7.928928,  0.014429,  0.009331],
        [7.932574,  0.012521,  0.009184],
        [7.950084,  0.010100,  0.009399],
        [7.951667,  0.009634,  0.009198],
        [7.960694,  0.006413,  0.009194],
        [7.960971,  0.002871,  0.008966],
        [7.964212, -0.002912,  0.008821],
        [7.967149, -0.008747,  0.008671],
        [7.983160, -0.012419,  0.008857],
        [7.977084, -0.013868,  0.008480],
        [8.002177, -0.016046,  0.008899],
        [8.000570, -0.012977,  0.008634],
        [8.007643, -0.013921,  0.008595],
        [8.013407, -0.014635,  0.008524],
        [8.032255, -0.013530,  0.008783],
        [8.044336, -0.008506,  0.008865],
        [8.051776, -0.004330,  0.008829],
        [8.064664, -0.001444,  0.008931],
        [8.084189,  0.003236,  0.009194],
        [8.084259,  0.007552,  0.008967],
        [8.085181,  0.005620,  0.008768],
        [8.096245,  0.002124,  0.008825],
        [8.090934, -0.001635,  0.008474],
        [8.099770, -0.007958,  0.008483],
        [8.096154, -0.013103,  0.008184],
        [8.099051, -0.019929,  0.008054],
        [8.084586, -0.028159,  0.007498],
        [8.104155, -0.036254,  0.007795],
        [8.120386, -0.034007,  0.008003],
        [8.127820, -0.028418,  0.007989],
        [8.141545, -0.022832,  0.008131],
        [8.140564, -0.018133,  0.007906],
        [8.143851, -0.017924,  0.007793],
        [8.154123, -0.017810,  0.007854],
        [8.166200, -0.015328,  0.007958],
        [8.178658, -0.011022,  0.008068],
        [8.187909, -0.006356,  0.008097],
        [8.184427, -0.004891,  0.007813],
        [8.193450, -0.007041,  0.007842],
        [8.222013, -0.004035,  0.008351],
        [8.222792,  0.002872,  0.008165],
        [8.232604,  0.004769,  0.008206],
        [8.232920,  0.004771,  0.008012],
        [8.237120,  0.001496,  0.007919],
        [8.246178, -0.001449,  0.007947],
        [8.250045, -0.003453,  0.007847],
        [8.256698, -0.005869,  0.007817],
        [8.236922, -0.012061,  0.007141],
        [8.247774, -0.022672,  0.007232],
        [8.269118, -0.024127,  0.007578],
        [8.277812, -0.019260,  0.007605],
        [8.271764, -0.017391,  0.007270],
        [8.279040, -0.019452,  0.007271],
        [8.266583, -0.023011,  0.006787],
        [8.262738, -0.031602,  0.006527],
        [8.272654, -0.037532,  0.006610],
        [8.270217, -0.039533,  0.006388],
        [8.273835, -0.041767,  0.006320],
        [8.280077, -0.041673,  0.006319],
        [8.301972, -0.036708,  0.006700],
        [8.307430, -0.027378,  0.006670],
        [8.317328, -0.020258,  0.006749],
        [8.329307, -0.013129,  0.006877],
        [8.336112, -0.006696,  0.006875],
        [8.338223, -0.003415,  0.006758],
        [8.344312, -0.002854,  0.006742],
        [8.350658, -0.002641,  0.006732],
        [8.358162, -0.002359,  0.006751],
        [8.369502, -0.001041,  0.006863],
        [8.373472,  0.000659,  0.006793],
        [8.385533,  0.001666,  0.006922],
        [8.384242,  0.002319,  0.006721],
        [8.391933,  0.000286,  0.006744],
        [8.396505, -0.001005,  0.006691],
        [8.404956, -0.002052,  0.006734],
        [8.416432, -0.001205,  0.006850],
        [8.424022,  0.000903,  0.006868],
        [8.436123,  0.003217,  0.006996],
        [8.440287,  0.005474,  0.006927],
        [8.450172,  0.006167,  0.007000],
        [8.455752,  0.006879,  0.006965],
        [8.465200,  0.006933,  0.007026],
        [8.472627,  0.007404,  0.007035],
        [8.477446,  0.007018,  0.006981],
        [8.479558,  0.004906,  0.006862],
        [8.486204,  0.001911,  0.006857],
        [8.496122,  0.000521,  0.006932],
        [8.500348,  0.000127,  0.006865],
        [8.500417, -0.002101,  0.006699],
        [8.497303, -0.007029,  0.006459],
        [8.498474, -0.013391,  0.006329],
        [8.507865, -0.017447,  0.006404],
        [8.511353, -0.018473,  0.006333],
        [8.512856, -0.019792,  0.006215],
        [8.521315, -0.020556,  0.006270],
        [8.526119, -0.019481,  0.006234],
        [8.533359, -0.018068,  0.006258],
        [8.544398, -0.015138,  0.006375],
        [8.544422, -0.012281,  0.006220],
        [8.550013, -0.011958,  0.006205],
        [8.556038, -0.011346,  0.006200],
        [8.569108, -0.009239,  0.006368],
        [8.573833, -0.005737,  0.006328],
        [8.581719, -0.003600,  0.006366],
        [8.589473, -0.001449,  0.006400],
        [8.599466,  0.000928,  0.006488],
        [8.604040,  0.003012,  0.006441],
        [8.607332,  0.002951,  0.006364],
        [8.618016,  0.002564,  0.006470]
    ]
}

"""
Clark's (1989) Bivariate Unobserved Components model of GDP

See http://econ.korea.ac.kr/~cjkim/MARKOV/prgmlist.htm#chapter3

Gauss Code: UC_BI.OPT
Datasets:   GDP4795.PRN, LHUR.PRN

"""
uc_bi = {
    'data': [
        [1239.5, 0],                [1247.199951, 0],
        [1255, 0],                  [1269.5, 0],
        [1284, 3.733333349],        [1295.699951, 3.666666667],
        [1303.800049, 3.766666651], [1316.400024, 3.833333333],
        [1305.300049, 4.666666667], [1302, 5.866666635],
        [1312.599976, 6.699999968], [1301.900024, 6.966666698],
        [1350.900024, 6.400000095], [1393.5, 5.566666762],
        [1445.199951, 4.633333365], [1484.5, 4.23333327],
        [1504.099976, 3.500000079], [1548.300049, 3.099999984],
        [1585.400024, 3.166666587], [1596, 3.366666635],
        [1607.699951, 3.066666683], [1612.099976, 2.966666698],
        [1621.900024, 3.233333349], [1657.800049, 2.833333333],
        [1687.300049, 2.699999968], [1695.300049, 2.566666683],
        [1687.900024, 2.733333349], [1671.199951, 3.699999968],
        [1660.800049, 5.266666571], [1658.400024, 5.800000032],
        [1677.699951, 5.966666698], [1698.300049, 5.333333333],
        [1742.5, 4.73333327],       [1758.599976, 4.399999936],
        [1778.199951, 4.099999905], [1793.900024, 4.23333327],
        [1787, 4.033333302],        [1798.5, 4.200000127],
        [1802.199951, 4.133333365], [1826.599976, 4.133333365],
        [1836.400024, 3.933333317], [1834.800049, 4.100000064],
        [1851.199951, 4.23333327],  [1830.5, 4.933333238],
        [1790.099976, 6.300000032], [1804.400024, 7.366666794],
        [1840.900024, 7.333333333], [1880.900024, 6.366666476],
        [1904.900024, 5.833333333], [1937.5, 5.099999905],
        [1930.800049, 5.266666571], [1941.900024, 5.600000064],
        [1976.900024, 5.133333365], [1971.699951, 5.23333327],
        [1973.699951, 5.533333302], [1961.099976, 6.266666571],
        [1977.400024, 6.800000032], [2006, 7],
        [2035.199951, 6.766666571], [2076.5, 6.199999968],
        [2103.800049, 5.633333365], [2125.699951, 5.533333302],
        [2142.600098, 5.566666603], [2140.199951, 5.533333302],
        [2170.899902, 5.766666571], [2199.5, 5.73333327],
        [2237.600098, 5.5],         [2254.5, 5.566666603],
        [2311.100098, 5.466666698], [2329.899902, 5.199999968],
        [2357.399902, 5],           [2364, 4.966666698],
        [2410.100098, 4.899999936], [2442.800049, 4.666666667],
        [2485.5, 4.366666794],      [2543.800049, 4.099999905],
        [2596.800049, 3.866666635], [2601.399902, 3.833333333],
        [2626.100098, 3.766666651], [2640.5, 3.699999968],
        [2657.199951, 3.833333333], [2669, 3.833333333],
        [2699.5, 3.799999952],      [2715.100098, 3.900000016],
        [2752.100098, 3.733333349], [2796.899902, 3.566666683],
        [2816.800049, 3.533333381], [2821.699951, 3.400000095],
        [2864.600098, 3.400000095], [2867.800049, 3.433333397],
        [2884.5, 3.566666683],      [2875.100098, 3.566666683],
        [2867.800049, 4.166666667], [2859.5, 4.76666673],
        [2895, 5.166666667],        [2873.300049, 5.833333333],
        [2939.899902, 5.933333397], [2944.199951, 5.900000095],
        [2962.300049, 6.033333302], [2977.300049, 5.933333397],
        [3037.300049, 5.76666673],  [3089.699951, 5.699999809],
        [3125.800049, 5.566666603], [3175.5, 5.366666635],
        [3253.300049, 4.933333397], [3267.600098, 4.933333397],
        [3264.300049, 4.800000191], [3289.100098, 4.76666673],
        [3259.399902, 5.133333206], [3267.600098, 5.199999968],
        [3239.100098, 5.633333365], [3226.399902, 6.599999905],
        [3154, 8.266667048],        [3190.399902, 8.866666794],
        [3249.899902, 8.46666654],  [3292.5, 8.299999873],
        [3356.699951, 7.73333327],  [3369.199951, 7.566666603],
        [3381, 7.733333429],        [3416.300049, 7.76666673],
        [3466.399902, 7.5],         [3525, 7.133333206],
        [3574.399902, 6.900000095], [3567.199951, 6.666666826],
        [3591.800049, 6.333333492], [3707, 6],
        [3735.600098, 6.033333302], [3779.600098, 5.900000095],
        [3780.800049, 5.866666794], [3784.300049, 5.699999968],
        [3807.5, 5.866666635],      [3814.600098, 5.966666698],
        [3830.800049, 6.300000191], [3732.600098, 7.333333333],
        [3733.5, 7.666666667],      [3808.5, 7.399999936],
        [3860.5, 7.433333397],      [3844.399902, 7.399999936],
        [3864.5, 7.399999936],      [3803.100098, 8.233333429],
        [3756.100098, 8.833333333], [3771.100098, 9.433333397],
        [3754.399902, 9.900000254], [3759.600098, 10.66666667],
        [3783.5, 10.36666648],      [3886.5, 10.13333352],
        [3944.399902, 9.366666476], [4012.100098, 8.53333346],
        [4089.5, 7.866666794],      [4144, 7.433333238],
        [4166.399902, 7.433333397], [4194.200195, 7.300000032],
        [4221.799805, 7.23333327],  [4254.799805, 7.300000032],
        [4309, 7.199999968],        [4333.5, 7.033333302],
        [4390.5, 7.033333143],      [4387.700195, 7.166666508],
        [4412.600098, 6.966666698], [4427.100098, 6.833333333],
        [4460, 6.599999905],        [4515.299805, 6.26666673],
        [4559.299805, 6],           [4625.5, 5.833333333],
        [4655.299805, 5.699999809], [4704.799805, 5.466666698],
        [4734.5, 5.466666698],      [4779.700195, 5.333333492],
        [4817.600098, 5.199999968], [4839, 5.23333327],
        [4839, 5.23333327],         [4856.700195, 5.366666794],
        [4898.299805, 5.300000032], [4917.100098, 5.333333333],
        [4906.5, 5.699999968],      [4867.200195, 6.133333365],
        [4842, 6.600000064],        [4867.899902, 6.833333333],
        [4879.899902, 6.866666794], [4880.799805, 7.100000064],
        [4918.5, 7.366666794],      [4947.5, 7.600000064],
        [4990.5, 7.633333206],      [5060.700195, 7.366666794],
        [5075.299805, 7.133333365], [5105.399902, 7.066666603],
        [5139.399902, 6.800000032], [5218, 6.633333365],
        [5261.100098, 6.633333206], [5314.100098, 6.199999968],
        [5367, 6],                  [5433.799805, 5.566666603],
        [5470.100098, 5.533333302], [5487.799805, 5.633333206],
        [5544.600098, 5.633333206]
    ],
    'start': 21-4-1,
    'parameters': [
        0.004863, 0.00668, 0.000295, 0.001518, 0.000306, 1.43859, -0.517385,
        -0.336789, -0.163511, -0.072012
    ],
    'standard_errors': [
        0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
        0.000000, 0.000000, 0.000000
    ],
    # See Notes in the file docstring for more information on the loglikelihood
    # 'loglike': 1566.994602,
    'loglike': 1485.41934,
    'states': [
        [7.359396,  0.023164,  0.013713,  0.044071],
        [7.362719,  0.022574,  0.012832,  0.043161],
        [7.373038,  0.018315,  0.012796,  0.044467],
        [7.393634,  0.019613,  0.012635,  0.037801],
        [7.408780,  0.022105,  0.012863,  0.038367],
        [7.414012,  0.021603,  0.012370,  0.038493],
        [7.414058,  0.017182,  0.011659,  0.039431],
        [7.416921,  0.004376,  0.011582,  0.044744],
        [7.429820, -0.014765,  0.012230,  0.051206],
        [7.435752, -0.022143,  0.011632,  0.048723],
        [7.447268, -0.022089,  0.011509,  0.047309],
        [7.450937, -0.013554,  0.010723,  0.043338],
        [7.467363, -0.004287,  0.010903,  0.041325],
        [7.472477, -0.000204,  0.010666,  0.042662],
        [7.480909,  0.002448,  0.010461,  0.041647],
        [7.491498,  0.000649,  0.010580,  0.043180],
        [7.487065,  0.001229,  0.009635,  0.041911],
        [7.495837, -0.001129,  0.009609,  0.042236],
        [7.497666, -0.000903,  0.009111,  0.041357],
        [7.510478, -0.000266,  0.009281,  0.040804],
        [7.513518,  0.002045,  0.008890,  0.040138],
        [7.516030, -0.001339,  0.008633,  0.041623],
        [7.526775, -0.003186,  0.008721,  0.041167],
        [7.526058, -0.013713,  0.008459,  0.045277],
        [7.524449, -0.034422,  0.008246,  0.050757],
        [7.544061, -0.046078,  0.008895,  0.051237],
        [7.558430, -0.040420,  0.009045,  0.048770],
        [7.563474, -0.023969,  0.008620,  0.045247],
        [7.568520, -0.016336,  0.008491,  0.046358],
        [7.575711, -0.006557,  0.008303,  0.044317],
        [7.576186, -0.010497,  0.008082,  0.047851],
        [7.587245, -0.015823,  0.008289,  0.048671],
        [7.596353, -0.007068,  0.008129,  0.045036],
        [7.595725, -0.009074,  0.007813,  0.047808],
        [7.602240, -0.014574,  0.007805,  0.048940],
        [7.606393, -0.025132,  0.007768,  0.051857],
        [7.620310, -0.030772,  0.008121,  0.052282],
        [7.634385, -0.030487,  0.008445,  0.052385],
        [7.643520, -0.025171,  0.008447,  0.051739],
        [7.653856, -0.015416,  0.008463,  0.050131],
        [7.658758, -0.007257,  0.008248,  0.049650],
        [7.668470, -0.006613,  0.008393,  0.050929],
        [7.677398, -0.007622,  0.008453,  0.051544],
        [7.676653, -0.007998,  0.007963,  0.051552],
        [7.693051, -0.010154,  0.008439,  0.052071],
        [7.704488, -0.008503,  0.008587,  0.051841],
        [7.717178, -0.004019,  0.008767,  0.051084],
        [7.726080, -0.005396,  0.008853,  0.052655],
        [7.747984, -0.002505,  0.009499,  0.051686],
        [7.752362,  0.001219,  0.009225,  0.051659],
        [7.761774,  0.003541,  0.009215,  0.051256],
        [7.765241,  0.002870,  0.008933,  0.051734],
        [7.782698,  0.004726,  0.009337,  0.050749],
        [7.792415,  0.008486,  0.009331,  0.050208],
        [7.805420,  0.012809,  0.009482,  0.049330],
        [7.824241,  0.017173,  0.009946,  0.048673],
        [7.841661,  0.020374,  0.010335,  0.048518],
        [7.845477,  0.018328,  0.010035,  0.049472],
        [7.855021,  0.018234,  0.009954,  0.048333],
        [7.860222,  0.018502,  0.009674,  0.047770],
        [7.869432,  0.015597,  0.009665,  0.048076],
        [7.874754,  0.014705,  0.009402,  0.047404],
        [7.885524,  0.015297,  0.009434,  0.046559],
        [7.893504,  0.013081,  0.009381,  0.047074],
        [7.904608,  0.015511,  0.009399,  0.045546],
        [7.917944,  0.018323,  0.009577,  0.044894],
        [7.925926,  0.017430,  0.009516,  0.045383],
        [7.927535,  0.017559,  0.009058,  0.044576],
        [7.942312,  0.017872,  0.009324,  0.043809],
        [7.945307,  0.015993,  0.009010,  0.044250],
        [7.954008,  0.013098,  0.008988,  0.044131],
        [7.952052,  0.011791,  0.008384,  0.043624],
        [7.959391,  0.001910,  0.008428,  0.045708],
        [7.965660, -0.007258,  0.008362,  0.046727],
        [7.981621, -0.010881,  0.008753,  0.046477],
        [7.984173, -0.020957,  0.008572,  0.049592],
        [8.005586, -0.019456,  0.009162,  0.047685],
        [8.005729, -0.018136,  0.008728,  0.048606],
        [8.014046, -0.020325,  0.008746,  0.049461],
        [8.017136, -0.018363,  0.008419,  0.048869],
        [8.032555, -0.013831,  0.008752,  0.048074],
        [8.047624, -0.011794,  0.009127,  0.048945],
        [8.057447, -0.010002,  0.009179,  0.049253],
        [8.069676, -0.006456,  0.009316,  0.048733],
        [8.086067,  0.001358,  0.009622,  0.047288],
        [8.091964, -0.000153,  0.009522,  0.049344],
        [8.091093, -0.000292,  0.008944,  0.048749],
        [8.097901,  0.000468,  0.008797,  0.048031],
        [8.095928, -0.006630,  0.008327,  0.050163],
        [8.099455, -0.007644,  0.008011,  0.048832],
        [8.097491, -0.014440,  0.007571,  0.050705],
        [8.107855, -0.028733,  0.007866,  0.053837],
        [8.110267, -0.053840,  0.007861,  0.059886],
        [8.125657, -0.057755,  0.008165,  0.057870],
        [8.132368, -0.045989,  0.007976,  0.055472],
        [8.141350, -0.041948,  0.008115,  0.057302],
        [8.151163, -0.032449,  0.008143,  0.055996],
        [8.152718, -0.030287,  0.007876,  0.057718],
        [8.159199, -0.033272,  0.007876,  0.059298],
        [8.168512, -0.032198,  0.007945,  0.059157],
        [8.177076, -0.026204,  0.007940,  0.058345],
        [8.187094, -0.019458,  0.008032,  0.057956],
        [8.197282, -0.015729,  0.008171,  0.058494],
        [8.193424, -0.013887,  0.007551,  0.058849],
        [8.195445, -0.009036,  0.007189,  0.057355],
        [8.219129, -0.001151,  0.007994,  0.055918],
        [8.228193, -0.002529,  0.008171,  0.058480],
        [8.238694, -0.001321,  0.008265,  0.057875],
        [8.239265, -0.001575,  0.007876,  0.058242],
        [8.238615,  0.000002,  0.007380,  0.057294],
        [8.247042, -0.002314,  0.007465,  0.057928],
        [8.250763, -0.004172,  0.007278,  0.058161],
        [8.259723, -0.008894,  0.007418,  0.059249],
        [8.251776, -0.026916,  0.006833,  0.064114],
        [8.255284, -0.030182,  0.006553,  0.061881],
        [8.266278, -0.021287,  0.006670,  0.059379],
        [8.278996, -0.020444,  0.007088,  0.061459],
        [8.276100, -0.021727,  0.006610,  0.062542],
        [8.280788, -0.021201,  0.006491,  0.062160],
        [8.278326, -0.034755,  0.006232,  0.066766],
        [8.274878, -0.043742,  0.005752,  0.067439],
        [8.284140, -0.049018,  0.005975,  0.068240],
        [8.284696, -0.054013,  0.005779,  0.070185],
        [8.295259, -0.063191,  0.006173,  0.073204],
        [8.294049, -0.055644,  0.005677,  0.070929],
        [8.312751, -0.047486,  0.006364,  0.070890],
        [8.314987, -0.034935,  0.006106,  0.070042],
        [8.319216, -0.022145,  0.005948,  0.068793],
        [8.328130, -0.011952,  0.006079,  0.068246],
        [8.335603, -0.006186,  0.006168,  0.068566],
        [8.342256, -0.007448,  0.006264,  0.070071],
        [8.347204, -0.005746,  0.006165,  0.069450],
        [8.352469, -0.004453,  0.006117,  0.069428],
        [8.361097, -0.005294,  0.006276,  0.069985],
        [8.371490, -0.003029,  0.006466,  0.069408],
        [8.374839, -0.000708,  0.006297,  0.069321],
        [8.387558, -0.000360,  0.006647,  0.069525],
        [8.389888, -0.003327,  0.006475,  0.070742],
        [8.392638, -0.000419,  0.006204,  0.069151],
        [8.393948,  0.001552,  0.005933,  0.068889],
        [8.398118,  0.004786,  0.005795,  0.067943],
        [8.405206,  0.010021,  0.005807,  0.066720],
        [8.411466,  0.013458,  0.005815,  0.066388],
        [8.423676,  0.015664,  0.006137,  0.066035],
        [8.429154,  0.016608,  0.006101,  0.066025],
        [8.436810,  0.019528,  0.006129,  0.064861],
        [8.443949,  0.018683,  0.006201,  0.065261],
        [8.452078,  0.020055,  0.006258,  0.064296],
        [8.458518,  0.021513,  0.006241,  0.063727],
        [8.464632,  0.019831,  0.006247,  0.063995],
        [8.466064,  0.018399,  0.005975,  0.063618],
        [8.472230,  0.015884,  0.005980,  0.063519],
        [8.479638,  0.017005,  0.006004,  0.062420],
        [8.484587,  0.015887,  0.005959,  0.062648],
        [8.489326,  0.008990,  0.005964,  0.064131],
        [8.488979,  0.001295,  0.005670,  0.064976],
        [8.490673, -0.005590,  0.005482,  0.065457],
        [8.497926, -0.007508,  0.005549,  0.064894],
        [8.500109, -0.007229,  0.005365,  0.064766],
        [8.503909, -0.010844,  0.005337,  0.065932],
        [8.514619, -0.013860,  0.005646,  0.066461],
        [8.522892, -0.016254,  0.005824,  0.067295],
        [8.530933, -0.015642,  0.005937,  0.067215],
        [8.539466, -0.010206,  0.006025,  0.066154],
        [8.539379, -0.007238,  0.005717,  0.066474],
        [8.544783, -0.006729,  0.005709,  0.066661],
        [8.547336, -0.002645,  0.005492,  0.065594],
        [8.559110,  0.000759,  0.005808,  0.065240],
        [8.567654,  0.000441,  0.005997,  0.066197],
        [8.571449,  0.006669,  0.005791,  0.064272],
        [8.578487,  0.009537,  0.005855,  0.064241],
        [8.584874,  0.015520,  0.005811,  0.062693],
        [8.592063,  0.014989,  0.005925,  0.063559],
        [8.598175,  0.012108,  0.005956,  0.064012],
        [8.608197,  0.012383,  0.006133,  0.063200]
    ]
}

"""
Kim and Nelson's (1989) time-varying parameters model of monetary growth

See http://econ.korea.ac.kr/~cjkim/MARKOV/prgmlist.htm#chapter3

Gauss Code: TVP.OPT
Datasets:   TVP.PRN

"""
tvp = {
    'data': [
        [59.300000,  0.751356,  0.226667,  0.191755,  0.103000,  0.828304],
        [59.400000, -0.869262,  0.540000,  0.496848,  0.044000,  0.751356],
        [60.100000, -0.259896,  0.690000,  0.608136,  0.058000, -0.869262],
        [60.200000, -0.165737, -0.356667,  0.113615,  0.188000, -0.259896],
        [60.300000,  0.943401, -0.880000,  0.603775,  0.126000, -0.165737],
        [60.400000, -0.046962, -0.633333,  0.037615,  0.087000,  0.943401],
        [61.100000,  0.538708, -0.053333,  0.637303,  0.096000, -0.046962],
        [61.200000,  0.767892,  0.043333,  0.223964, -0.004000,  0.538708],
        [61.300000,  0.600882, -0.046667, -0.037292, -0.043000,  0.767892],
        [61.400000,  0.940261,  0.000000,  0.409455, -0.034000,  0.600882],
        [62.100000,  0.637094,  0.156667,  0.111379, -0.059000,  0.940261],
        [62.200000,  0.700645,  0.263333,  0.407333, -0.201000,  0.637094],
        [62.300000, -0.112678, -0.010000,  0.368868, -0.169000,  0.700645],
        [62.400000,  0.562120,  0.126667,  0.257401, -0.145000, -0.112678],
        [63.100000,  1.026110, -0.026667,  0.256739, -0.122000,  0.562120],
        [63.200000,  0.949756,  0.093333,  0.329128, -0.070000,  1.026110],
        [63.300000,  1.006140,  0.030000,  0.182382,  0.029000,  0.949756],
        [63.400000,  0.953003,  0.356667,  0.581397, -0.045000,  1.006140],
        [64.100000,  0.687438,  0.203333,  0.289436, -0.076000,  0.953003],
        [64.200000,  0.704003,  0.033333,  0.432589, -0.225000,  0.687438],
        [64.300000,  1.623650, -0.053333,  0.143781, -0.337000,  0.704003],
        [64.400000,  1.288460,  0.020000,  0.179437, -0.220000,  1.623650],
        [65.100000,  0.740592,  0.186667,  0.500716, -0.151000,  1.288460],
        [65.200000,  0.592627,  0.206667,  0.320570, -0.051000,  0.740592],
        [65.300000,  1.154680, -0.016667,  0.638074, -0.105000,  0.592627],
        [65.400000,  1.816230, -0.006667,  0.282287, -0.348000,  1.154680],
        [66.100000,  1.706090,  0.300000,  0.527149, -0.441000,  1.816230],
        [66.200000,  1.063740,  0.443333,  0.941922, -0.397000,  1.706090],
        [66.300000, -0.288990, -0.023333,  0.898727, -0.352000,  1.063740],
        [66.400000,  0.269746,  0.456667,  0.856609, -0.492000, -0.288990],
        [67.100000,  1.014660,  0.166667,  0.815499, -0.547000,  0.269746],
        [67.200000,  1.399680, -0.696667,  0.337838, -0.715000,  1.014660],
        [67.300000,  2.192170, -0.853333,  0.538179, -0.707000,  1.399680],
        [67.400000,  1.586030,  0.640000,  1.034550, -0.749000,  2.192170],
        [68.100000,  1.347370,  0.453333,  0.958213, -0.709000,  1.586030],
        [68.200000,  1.698550,  0.296667,  1.079320, -0.641000,  1.347370],
        [68.300000,  1.928650,  0.470000,  0.971196, -0.763000,  1.698550],
        [68.400000,  2.111430, -0.323333,  1.280420, -0.520000,  1.928650],
        [69.100000,  1.786810,  0.390000,  1.232830, -0.389000,  2.111430],
        [69.200000,  0.807717,  0.506667,  1.217810, -0.147000,  1.786810],
        [69.300000,  0.425952,  0.103333,  1.509340, -0.096000,  0.807717],
        [69.400000,  0.765415,  0.826667,  1.456780, -0.184000,  0.425952],
        [70.100000,  1.048990,  0.330000,  1.435860, -0.136000,  0.765415],
        [70.200000,  0.735771, -0.143333,  1.620310, -0.124000,  1.048990],
        [70.300000,  1.345490, -0.533333,  1.392940, -0.331000,  0.735771],
        [70.400000,  1.776580, -0.346667,  1.117650, -0.380000,  1.345490],
        [71.100000,  1.715200, -0.976667,  1.386750, -0.369000,  1.776580],
        [71.200000,  2.103440, -1.513330,  0.839635, -0.421000,  1.715200],
        [71.300000,  1.709950,  0.410000,  0.943139, -0.474000,  2.103440],
        [71.400000,  0.945527,  0.760000,  1.016360, -0.444000,  1.709950],
        [72.100000,  1.935680, -0.780000,  0.653775, -0.354000,  0.945527],
        [72.200000,  1.633840, -0.793333,  0.892018, -0.213000,  1.935680],
        [72.300000,  2.046530,  0.333333,  0.590286, -0.448000,  1.633840],
        [72.400000,  2.460580,  0.450000,  0.878951, -0.242000,  2.046530],
        [73.100000,  1.996500,  0.643333,  1.002650, -0.614000,  2.460580],
        [73.200000,  1.171280,  0.836667,  1.511230, -0.421000,  1.996500],
        [73.300000,  1.183290,  0.903333,  2.022340, -0.393000,  1.171280],
        [73.400000,  1.245250,  1.720000,  2.031930, -0.234000,  1.183290],
        [74.100000,  1.665560, -0.823333,  2.452910, -0.238000,  1.245250],
        [74.200000,  0.890214,  0.116667,  2.936820, -0.210000,  1.665560],
        [74.300000,  0.894558,  0.536667,  2.601170, -0.253000,  0.890214],
        [74.400000,  1.200520,  0.036667,  2.869930, -0.062000,  0.894558],
        [75.100000,  0.708670, -0.830000,  3.049490, -0.199000,  1.200520],
        [75.200000,  1.543970, -1.610000,  2.053110, -0.328000,  0.708670],
        [75.300000,  1.833420, -0.356667,  1.220290, -1.242000,  1.543970],
        [75.400000,  0.806734,  0.936667,  1.987640, -0.643000,  1.833420],
        [76.100000,  1.334040, -0.703333,  1.848330, -0.680000,  0.806734],
        [76.200000,  1.628970, -0.710000,  1.140580, -0.567000,  1.334040],
        [76.300000,  1.064080,  0.240000,  0.832182, -0.469000,  1.628970],
        [76.400000,  1.943510, -0.006667,  1.546780, -0.528000,  1.064080],
        [77.100000,  2.298210, -0.476667,  1.427490, -0.561000,  1.943510],
        [77.200000,  1.718220, -0.043333,  1.840840, -0.331000,  2.298210],
        [77.300000,  1.689200,  0.210000,  1.770610, -0.441000,  1.718220],
        [77.400000,  2.092150,  0.656667,  1.376040, -0.682000,  1.689200],
        [78.100000,  1.921800,  0.613333,  1.447210, -0.601000,  2.092150],
        [78.200000,  2.193200,  0.283333,  1.691790, -0.572000,  1.921800],
        [78.300000,  2.023920,  0.083333,  2.286910, -0.440000,  2.193200],
        [78.400000,  1.771440,  0.836667,  2.269530, -0.429000,  2.023920],
        [79.100000,  1.221600,  1.256670,  2.318120, -0.417000,  1.771440],
        [79.200000,  1.827280,  0.813333,  2.458800, -0.223000,  1.221600],
        [79.300000,  3.401510, -0.006667,  3.197490, -0.123000,  1.827280],
        [79.400000,  1.120030,  0.296667,  3.143710, -0.306000,  3.401510],
        [80.100000,  1.082200,  2.170000,  3.121070, -0.335000,  1.120030],
        [80.200000, -1.494280,  1.510000,  3.860710, -0.435000,  1.082200],
        [80.300000,  4.787210, -3.736670,  3.323020, -0.454000, -1.494280],
        [80.400000,  2.670280, -0.463333,  1.835160, -0.458000,  4.787210],
        [81.100000,  0.430245,  4.460000,  2.795170, -0.398000,  2.670280],
        [81.200000,  1.647980,  0.776667,  2.693850, -0.226000,  0.430245],
        [81.300000,  1.652030,  0.516667,  2.066840, -0.169000,  1.647980],
        [81.400000,  1.230860,  0.146666,  2.751290, -0.274000,  1.652030],
        [82.100000,  1.807140, -3.303330,  1.615160, -0.440000,  1.230860],
        [82.200000,  0.417009,  1.063330,  0.897925, -0.351000,  1.807140],
        [82.300000,  2.132120, -0.393333,  1.401560, -0.311000,  0.417009],
        [82.400000,  3.936900, -3.103330,  1.736170, -0.630000,  2.132120],
        [83.100000,  2.534230, -1.410000,  0.318655, -0.917000,  3.936900],
        [83.200000,  2.790380,  0.200000,  0.102209, -0.838000,  2.534230],
        [83.300000,  2.733980,  0.290000,  1.151120, -0.803000,  2.790380],
        [83.400000,  1.612830,  0.743333,  0.993698, -0.966000,  2.733980],
        [84.100000,  1.537230, -0.340000,  0.950915, -1.000000,  1.612830],
        [84.200000,  1.581670,  0.370000,  1.290200, -1.044000,  1.537230],
        [84.300000,  1.071160,  0.626667,  0.994496, -1.210000,  1.581670],
        [84.400000,  1.059810,  0.523333,  0.888801, -1.295000,  1.071160],
        [85.100000,  2.663020, -1.516670,  0.817542, -1.419000,  1.059810],
        [85.200000,  2.571070, -0.620000,  0.821403, -1.217000,  2.663020],
        [85.300000,  3.494150, -0.723333,  1.115930, -1.675000,  2.571070],
        [85.400000,  2.674520, -0.353333,  0.579113, -1.619000,  3.494150]
    ],
    'start': 11-1,
    'parameters': [
        0.3712, 0.1112, 0.0171, 0.2720, 0.0378, 0.0224
    ],
    'standard_errors': [
        0.0632, 0.0627, 0.0342, 0.0607, 0.1642, 0.0374
    ],
    'loglike': -97.092423,
    'states': [
        [0.4965, -0.6580,  0.5813, -3.2549, -0.0724],
        [0.5126, -0.6555,  0.5719, -3.3179, -0.0709],
        [0.4389, -0.5831,  0.4181, -2.1186, -0.0189],
        [0.3019, -0.3959,  0.1196, -0.7239,  0.0207],
        [0.3727, -0.4049,  0.0580, -0.9298, -0.0335],
        [0.4829, -0.4390,  0.1610, -1.0697, -0.0215],
        [0.5108, -0.3993,  0.4471, -0.6918,  0.0572],
        [0.6079, -0.3483,  0.4996, -0.1870,  0.0972],
        [0.6023, -0.3410,  0.5741, -0.1433,  0.1038],
        [0.5729, -0.3442,  0.5667, -0.1220,  0.1003],
        [0.5521, -0.3105,  0.4848,  0.1181,  0.1097],
        [0.7521, -0.4232,  0.0775, -1.1218,  0.0736],
        [0.7626, -0.4281,  0.0718, -1.1584,  0.1052],
        [0.7570, -0.4325, -0.1868, -1.2219,  0.0883],
        [0.7035, -0.4465, -0.2159, -1.3017,  0.0980],
        [0.7138, -0.4792,  0.1306, -1.2976,  0.0794],
        [0.7741, -0.5153,  0.1357, -1.7173,  0.0976],
        [0.7618, -0.5138,  0.1794, -1.7576,  0.1067],
        [0.8309, -0.5206, -0.2358, -1.7412,  0.0927],
        [0.7468, -0.3393, -1.2197, -1.5994,  0.1837],
        [0.7451, -0.3401, -1.2230, -1.5955,  0.1856],
        [0.7666, -0.3655, -1.0822, -1.7327,  0.1726],
        [0.7155, -0.2192, -0.9440, -1.2904,  0.1621],
        [0.6831, -0.3288, -0.7068, -1.4245,  0.2053],
        [0.6554, -0.2914, -0.5176, -1.4186,  0.2319],
        [0.6518, -0.2933, -0.5437, -1.4185,  0.2322],
        [0.6797, -0.3100, -0.2444, -1.3599,  0.2240],
        [0.6891, -0.3016, -0.1704, -1.3899,  0.2279],
        [0.6832, -0.3461,  0.0626, -1.2895,  0.2391],
        [0.7040, -0.3385,  0.1256, -1.2539,  0.2443],
        [0.6050, -0.3543, -0.0651, -1.4003,  0.2434],
        [0.5992, -0.3375, -0.2575, -1.4240,  0.2665],
        [0.6359, -0.3069, -0.0895, -1.4528,  0.2445],
        [0.6479, -0.3149,  0.0208, -1.4431,  0.2468],
        [0.6738, -0.2881, -0.1767, -1.4472,  0.2379],
        [0.6790, -0.2942, -0.1342, -1.4632,  0.2359],
        [0.7273, -0.2934, -0.0569, -1.4789,  0.2475],
        [0.7441, -0.2804, -0.1501, -1.4776,  0.2392],
        [0.7523, -0.2836, -0.1486, -1.4775,  0.2402],
        [0.7538, -0.2824, -0.1460, -1.4781,  0.2406],
        [0.6948, -0.3274, -0.3619, -1.4639,  0.2477],
        [0.8999, -0.3630, -0.3050, -1.4275,  0.2248],
        [0.8989, -0.3619, -0.3151, -1.4319,  0.2236],
        [1.0628, -0.3234, -0.3110, -1.4815,  0.2314],
        [1.0959, -0.2846,  0.2317, -1.3609,  0.2623],
        [1.2113, -0.2943, -0.1348, -1.1898,  0.2490],
        [1.3005, -0.2927, -0.5252, -1.1919,  0.2722],
        [1.2629, -0.2966, -0.3244, -1.1993,  0.2518],
        [1.2894, -0.2782, -0.1285, -1.1538,  0.2469],
        [1.2988, -0.2701, -0.1707, -1.1522,  0.2462],
        [1.3264, -0.2744, -0.3310, -1.1545,  0.2403],
        [1.3314, -0.2734, -0.3105, -1.1558,  0.2387],
        [1.3363, -0.2779, -0.1683, -1.1320,  0.2390],
        [1.3662, -0.2609, -0.4110, -1.0999,  0.2321],
        [1.3796, -0.2636, -0.3915, -1.1038,  0.2307],
        [1.3209, -0.2770, -0.5148, -0.7347,  0.2208],
        [1.3205, -0.2821, -0.5581, -0.7462,  0.2182],
        [1.3302, -0.2880, -0.4981, -0.7492,  0.2119],
        [1.3132, -0.2871, -0.5063, -0.7533,  0.2103],
        [1.1144, -0.3065, -0.5572, -0.8137,  0.1905],
        [1.0196, -0.3003,  0.0227, -0.8400,  0.1677],
        [1.0078, -0.3079,  0.1474, -0.8392,  0.1853],
        [1.0269, -0.3099,  0.0285, -0.8580,  0.1771],
        [1.0267, -0.3100,  0.0262, -0.8578,  0.1772],
        [1.0764, -0.2997,  0.1328, -0.8925,  0.1745],
        [1.0736, -0.3001,  0.1081, -0.8929,  0.1726],
        [1.0678, -0.3024,  0.1891, -0.8931,  0.1719],
        [1.0911, -0.3000,  0.0994, -0.8979,  0.1711],
        [1.0908, -0.3005,  0.0906, -0.8981,  0.1713],
        [1.0801, -0.3069, -0.0355, -0.8981,  0.1765],
        [1.1085, -0.3117,  0.1850, -0.8749,  0.1620],
        [0.9984, -0.3376,  0.5659, -0.8696,  0.1808],
        [1.1165, -0.3357, -0.0249, -0.8350,  0.0692],
        [1.1387, -0.3214,  0.0824, -0.8444,  0.0475],
        [1.2924, -0.2922, -0.6439, -0.7666,  0.0701],
        [1.5698, -0.5257,  0.1098, -0.8641, -0.1011],
        [1.6453, -0.5272,  0.2241, -0.8317, -0.0215],
        [1.6441, -0.5139,  0.2853, -0.8310, -0.0269],
        [1.5947, -0.5017,  0.1260, -0.8471, -0.0137],
        [1.5838, -0.5013,  0.1080, -0.8505, -0.0145],
        [1.6726, -0.4944, -0.1613, -0.8227, -0.0206],
        [1.2829, -0.4045, -0.4496, -0.7630, -0.0195],
        [1.1279, -0.4203, -0.4611, -0.7772, -0.0174],
        [1.2281, -0.4152,  0.1011, -0.7625, -0.0589],
        [1.0691, -0.4452,  0.4701, -0.8390, -0.0221],
        [1.0707, -0.4453,  0.4689, -0.8399, -0.0217],
        [1.6109, -0.4191,  0.2096, -0.9584, -0.0488],
        [1.6012, -0.4095,  0.4301, -0.9771, -0.0446],
        [1.5902, -0.4308,  0.0350, -0.8648, -0.0457],
        [1.4448, -0.4179, -0.3688, -0.7584, -0.0026],
        [1.4450, -0.4180, -0.3791, -0.7575, -0.0023],
        [1.4060, -0.4274, -0.5805, -0.6579,  0.0065],
        [1.3527, -0.4308, -0.7075, -0.5849,  0.0222],
        [1.3832, -0.4496, -0.6074, -0.6392,  0.0158],
        [1.4081, -0.4536, -0.3465, -0.6110,  0.0586],
        [1.2291, -0.4531,  0.1552, -0.7252,  0.0818]
    ]
}

"""
Real GDP, 1947:1 - 2014:1

See http://research.stlouisfed.org/fred2/series/GDPC1

Stata Code: test_kalman_filter_gdp.do
R Code:     test_kalman_filter_gdp.R

"""
gdp = {
    'stata_params': [
        0.40725515, 0.18782621, -0.01514009, -0.01027267, -0.03642297,
        0.11576416, 0.02573029, -.00766572, 0.13506498, 0.08649569,
        0.06942822, -0.10685783, 0.00008
    ],
    'stata_loglike': 883.1262237746861
}
