#!/usr/bin/env python
class ProgressIndicator:
    def __init__(self):
        self.total_steps = 0
        self.current_step = 0
        self.current_step_caption = None
        self.total_substeps = 0
        self.current_substep = 0
        self.current_substep_caption = None
        self.show_caption = True
        self.prefix_str = '['
        self.suffix_str = '] '
        self.indicator_width = 40
        self.max_line_width = 80
        self.printer = None

    def CalculatePercentage(self):
        ts = self.total_steps
        cs = self.current_step
        tss = self.total_substeps
        css = self.current_substep
        if (cs > ts) or (ts <= 0):
            return 100.0
        elif (cs <= 0):
            return 0.0
        perstep = 100.0 / ts
        ret = perstep * (cs - 1)
        if (css > tss) or (tss <= 0):
            ret += perstep
        elif (css > 0):
            persubstep = perstep / tss
            ret += css * persubstep
        return max(min(ret, 100.0), 0.0)

    def GenerateProgressBar(self):
        percentage = self.CalculatePercentage()
        percent_str = f'{int(percentage)}%'
        used_length = len(percent_str)
        remain_length = max(self.indicator_width - used_length, 0)
        bar_width = max(min(int(remain_length * percentage / 100.0), remain_length), 0)
        empty_width = remain_length - bar_width
        return ('=' * bar_width) + percent_str + (' ' * empty_width)

    def GenerateCaption(self):
        if not self.show_caption:
            return ""
        used_length = self.indicator_width + len(self.prefix_str) + len(self.suffix_str)
        remain_length = max(self.max_line_width - used_length, 9) # Minimal string: "A...:B..."
        caption = ""
        if self.current_step_caption:
            caption = self.current_step_caption
        if (self.total_substeps > 1) and self.current_substep_caption:
            if caption:
                substep_len = len(self.current_substep_caption)
                if (len(caption) + substep_len > remain_length):
                    step_len = max(remain_length - substep_len, 1)
                    caption = caption[0:step_len] + '...'
                caption += ':'
            caption += self.current_substep_caption
        if (len(caption) > remain_length):
            caption = caption[0:remain_length - 3] + '...'
        return caption

    def GenerateProgressLine(self):
        return self.prefix_str + self.GenerateProgressBar() + self.suffix_str + self.GenerateCaption()

    def NewlinePrinter(self):
        print(self.GenerateProgressLine())

    def SameLinePrinter(self):
        print('\033[2K'+self.GenerateProgressLine(), flush=True)
        if 100.0 != self.CalculatePercentage():
            print('\033[1A', end='')

    def Print(self):
        if self.printer:
            self.printer()

    def Initialize(self, step_count):
        self.total_steps = step_count
        self.current_step = 0
        self.current_step_caption = None
        self.current_substep_caption = None

    def StartStep(self, substep_count=0, caption=None):
        self.current_step += 1
        self.current_step_caption = caption
        self.total_substeps = substep_count
        self.current_substep = 0
        self.current_substep_caption = None
        self.Print()

    def StartSubstep(self, caption=None, step_caption=None):
        self.current_substep += 1
        self.current_substep_caption = caption
        if step_caption:
            self.current_step_caption = step_caption
        self.Print()

    def Finish(self):
        self.current_step = self.total_steps
        self.current_step_caption = "Done"
        self.current_substep = self.total_substeps
        self.current_substep_caption = None
        self.Print()

g_progress_indicator = ProgressIndicator()
