﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/xray/XRayRequest.h>
#include <aws/xray/XRay_EXPORTS.h>

#include <utility>

namespace Aws {
namespace XRay {
namespace Model {

/**
 */
class DeleteResourcePolicyRequest : public XRayRequest {
 public:
  AWS_XRAY_API DeleteResourcePolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteResourcePolicy"; }

  AWS_XRAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the resource policy to delete.</p>
   */
  inline const Aws::String& GetPolicyName() const { return m_policyName; }
  inline bool PolicyNameHasBeenSet() const { return m_policyNameHasBeenSet; }
  template <typename PolicyNameT = Aws::String>
  void SetPolicyName(PolicyNameT&& value) {
    m_policyNameHasBeenSet = true;
    m_policyName = std::forward<PolicyNameT>(value);
  }
  template <typename PolicyNameT = Aws::String>
  DeleteResourcePolicyRequest& WithPolicyName(PolicyNameT&& value) {
    SetPolicyName(std::forward<PolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a specific policy revision to delete. Provide a
   * <code>PolicyRevisionId</code> to ensure an atomic delete operation. If the
   * provided revision id does not match the latest policy revision id, an
   * <code>InvalidPolicyRevisionIdException</code> exception is returned. </p>
   */
  inline const Aws::String& GetPolicyRevisionId() const { return m_policyRevisionId; }
  inline bool PolicyRevisionIdHasBeenSet() const { return m_policyRevisionIdHasBeenSet; }
  template <typename PolicyRevisionIdT = Aws::String>
  void SetPolicyRevisionId(PolicyRevisionIdT&& value) {
    m_policyRevisionIdHasBeenSet = true;
    m_policyRevisionId = std::forward<PolicyRevisionIdT>(value);
  }
  template <typename PolicyRevisionIdT = Aws::String>
  DeleteResourcePolicyRequest& WithPolicyRevisionId(PolicyRevisionIdT&& value) {
    SetPolicyRevisionId(std::forward<PolicyRevisionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_policyName;

  Aws::String m_policyRevisionId;
  bool m_policyNameHasBeenSet = false;
  bool m_policyRevisionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace XRay
}  // namespace Aws
