﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/qconnect/QConnectRequest.h>
#include <aws/qconnect/QConnect_EXPORTS.h>
#include <aws/qconnect/model/AIPromptInferenceConfiguration.h>
#include <aws/qconnect/model/AIPromptTemplateConfiguration.h>
#include <aws/qconnect/model/VisibilityStatus.h>

#include <utility>

namespace Aws {
namespace QConnect {
namespace Model {

/**
 */
class UpdateAIPromptRequest : public QConnectRequest {
 public:
  AWS_QCONNECT_API UpdateAIPromptRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateAIPrompt"; }

  AWS_QCONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If not provided, the Amazon Web Services SDK
   * populates this field. For more information about idempotency, see <a
   * href="http://aws.amazon.com/builders-library/making-retries-safe-with-idempotent-APIs/">Making
   * retries safe with idempotent APIs</a>..</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateAIPromptRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Q in Connect assistant. Can be either the ID or
   * the ARN. URLs cannot contain the ARN.</p>
   */
  inline const Aws::String& GetAssistantId() const { return m_assistantId; }
  inline bool AssistantIdHasBeenSet() const { return m_assistantIdHasBeenSet; }
  template <typename AssistantIdT = Aws::String>
  void SetAssistantId(AssistantIdT&& value) {
    m_assistantIdHasBeenSet = true;
    m_assistantId = std::forward<AssistantIdT>(value);
  }
  template <typename AssistantIdT = Aws::String>
  UpdateAIPromptRequest& WithAssistantId(AssistantIdT&& value) {
    SetAssistantId(std::forward<AssistantIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Q in Connect AI Prompt.</p>
   */
  inline const Aws::String& GetAiPromptId() const { return m_aiPromptId; }
  inline bool AiPromptIdHasBeenSet() const { return m_aiPromptIdHasBeenSet; }
  template <typename AiPromptIdT = Aws::String>
  void SetAiPromptId(AiPromptIdT&& value) {
    m_aiPromptIdHasBeenSet = true;
    m_aiPromptId = std::forward<AiPromptIdT>(value);
  }
  template <typename AiPromptIdT = Aws::String>
  UpdateAIPromptRequest& WithAiPromptId(AiPromptIdT&& value) {
    SetAiPromptId(std::forward<AiPromptIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The visibility status of the Amazon Q in Connect AI prompt.</p>
   */
  inline VisibilityStatus GetVisibilityStatus() const { return m_visibilityStatus; }
  inline bool VisibilityStatusHasBeenSet() const { return m_visibilityStatusHasBeenSet; }
  inline void SetVisibilityStatus(VisibilityStatus value) {
    m_visibilityStatusHasBeenSet = true;
    m_visibilityStatus = value;
  }
  inline UpdateAIPromptRequest& WithVisibilityStatus(VisibilityStatus value) {
    SetVisibilityStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration of the prompt template for this AI Prompt.</p>
   */
  inline const AIPromptTemplateConfiguration& GetTemplateConfiguration() const { return m_templateConfiguration; }
  inline bool TemplateConfigurationHasBeenSet() const { return m_templateConfigurationHasBeenSet; }
  template <typename TemplateConfigurationT = AIPromptTemplateConfiguration>
  void SetTemplateConfiguration(TemplateConfigurationT&& value) {
    m_templateConfigurationHasBeenSet = true;
    m_templateConfiguration = std::forward<TemplateConfigurationT>(value);
  }
  template <typename TemplateConfigurationT = AIPromptTemplateConfiguration>
  UpdateAIPromptRequest& WithTemplateConfiguration(TemplateConfigurationT&& value) {
    SetTemplateConfiguration(std::forward<TemplateConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the Amazon Q in Connect AI Prompt.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateAIPromptRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the model used for this AI Prompt.</p>  <p>For
   * information about which models are supported in each Amazon Web Services Region,
   * see <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/create-ai-prompts.html#cli-create-aiprompt">Supported
   * models for system/custom prompts</a>.</p>
   */
  inline const Aws::String& GetModelId() const { return m_modelId; }
  inline bool ModelIdHasBeenSet() const { return m_modelIdHasBeenSet; }
  template <typename ModelIdT = Aws::String>
  void SetModelId(ModelIdT&& value) {
    m_modelIdHasBeenSet = true;
    m_modelId = std::forward<ModelIdT>(value);
  }
  template <typename ModelIdT = Aws::String>
  UpdateAIPromptRequest& WithModelId(ModelIdT&& value) {
    SetModelId(std::forward<ModelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated inference configuration for the AI Prompt.</p>
   */
  inline const AIPromptInferenceConfiguration& GetInferenceConfiguration() const { return m_inferenceConfiguration; }
  inline bool InferenceConfigurationHasBeenSet() const { return m_inferenceConfigurationHasBeenSet; }
  template <typename InferenceConfigurationT = AIPromptInferenceConfiguration>
  void SetInferenceConfiguration(InferenceConfigurationT&& value) {
    m_inferenceConfigurationHasBeenSet = true;
    m_inferenceConfiguration = std::forward<InferenceConfigurationT>(value);
  }
  template <typename InferenceConfigurationT = AIPromptInferenceConfiguration>
  UpdateAIPromptRequest& WithInferenceConfiguration(InferenceConfigurationT&& value) {
    SetInferenceConfiguration(std::forward<InferenceConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_assistantId;

  Aws::String m_aiPromptId;

  VisibilityStatus m_visibilityStatus{VisibilityStatus::NOT_SET};

  AIPromptTemplateConfiguration m_templateConfiguration;

  Aws::String m_description;

  Aws::String m_modelId;

  AIPromptInferenceConfiguration m_inferenceConfiguration;
  bool m_clientTokenHasBeenSet = true;
  bool m_assistantIdHasBeenSet = false;
  bool m_aiPromptIdHasBeenSet = false;
  bool m_visibilityStatusHasBeenSet = false;
  bool m_templateConfigurationHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_modelIdHasBeenSet = false;
  bool m_inferenceConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace QConnect
}  // namespace Aws
