﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Document.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/opensearchserverless/OpenSearchServerlessRequest.h>
#include <aws/opensearchserverless/OpenSearchServerless_EXPORTS.h>

#include <utility>

namespace Aws {
namespace OpenSearchServerless {
namespace Model {

/**
 */
class CreateIndexRequest : public OpenSearchServerlessRequest {
 public:
  AWS_OPENSEARCHSERVERLESS_API CreateIndexRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateIndex"; }

  AWS_OPENSEARCHSERVERLESS_API Aws::String SerializePayload() const override;

  AWS_OPENSEARCHSERVERLESS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The unique identifier of the collection in which to create the index.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  CreateIndexRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the index to create. Index names must be lowercase and can't
   * begin with underscores (_) or hyphens (-).</p>
   */
  inline const Aws::String& GetIndexName() const { return m_indexName; }
  inline bool IndexNameHasBeenSet() const { return m_indexNameHasBeenSet; }
  template <typename IndexNameT = Aws::String>
  void SetIndexName(IndexNameT&& value) {
    m_indexNameHasBeenSet = true;
    m_indexName = std::forward<IndexNameT>(value);
  }
  template <typename IndexNameT = Aws::String>
  CreateIndexRequest& WithIndexName(IndexNameT&& value) {
    SetIndexName(std::forward<IndexNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The JSON schema definition for the index, including field mappings and
   * settings.</p>
   */
  inline Aws::Utils::DocumentView GetIndexSchema() const { return m_indexSchema; }
  inline bool IndexSchemaHasBeenSet() const { return m_indexSchemaHasBeenSet; }
  template <typename IndexSchemaT = Aws::Utils::Document>
  void SetIndexSchema(IndexSchemaT&& value) {
    m_indexSchemaHasBeenSet = true;
    m_indexSchema = std::forward<IndexSchemaT>(value);
  }
  template <typename IndexSchemaT = Aws::Utils::Document>
  CreateIndexRequest& WithIndexSchema(IndexSchemaT&& value) {
    SetIndexSchema(std::forward<IndexSchemaT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_indexName;

  Aws::Utils::Document m_indexSchema;
  bool m_idHasBeenSet = false;
  bool m_indexNameHasBeenSet = false;
  bool m_indexSchemaHasBeenSet = false;
};

}  // namespace Model
}  // namespace OpenSearchServerless
}  // namespace Aws
