﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lightsail/LightsailRequest.h>
#include <aws/lightsail/Lightsail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Lightsail {
namespace Model {

/**
 */
class GetRelationalDatabaseLogEventsRequest : public LightsailRequest {
 public:
  AWS_LIGHTSAIL_API GetRelationalDatabaseLogEventsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetRelationalDatabaseLogEvents"; }

  AWS_LIGHTSAIL_API Aws::String SerializePayload() const override;

  AWS_LIGHTSAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of your database for which to get log events.</p>
   */
  inline const Aws::String& GetRelationalDatabaseName() const { return m_relationalDatabaseName; }
  inline bool RelationalDatabaseNameHasBeenSet() const { return m_relationalDatabaseNameHasBeenSet; }
  template <typename RelationalDatabaseNameT = Aws::String>
  void SetRelationalDatabaseName(RelationalDatabaseNameT&& value) {
    m_relationalDatabaseNameHasBeenSet = true;
    m_relationalDatabaseName = std::forward<RelationalDatabaseNameT>(value);
  }
  template <typename RelationalDatabaseNameT = Aws::String>
  GetRelationalDatabaseLogEventsRequest& WithRelationalDatabaseName(RelationalDatabaseNameT&& value) {
    SetRelationalDatabaseName(std::forward<RelationalDatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the log stream.</p> <p>Use the <code>get relational database log
   * streams</code> operation to get a list of available log streams.</p>
   */
  inline const Aws::String& GetLogStreamName() const { return m_logStreamName; }
  inline bool LogStreamNameHasBeenSet() const { return m_logStreamNameHasBeenSet; }
  template <typename LogStreamNameT = Aws::String>
  void SetLogStreamName(LogStreamNameT&& value) {
    m_logStreamNameHasBeenSet = true;
    m_logStreamName = std::forward<LogStreamNameT>(value);
  }
  template <typename LogStreamNameT = Aws::String>
  GetRelationalDatabaseLogEventsRequest& WithLogStreamName(LogStreamNameT&& value) {
    SetLogStreamName(std::forward<LogStreamNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start of the time interval from which to get log events.</p>
   * <p>Constraints:</p> <ul> <li> <p>Specified in Coordinated Universal Time
   * (UTC).</p> </li> <li> <p>Specified in the Unix time format.</p> <p>For example,
   * if you wish to use a start time of October 1, 2018, at 8 PM UTC, then you input
   * <code>1538424000</code> as the start time.</p> </li> </ul>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  GetRelationalDatabaseLogEventsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end of the time interval from which to get log events.</p>
   * <p>Constraints:</p> <ul> <li> <p>Specified in Coordinated Universal Time
   * (UTC).</p> </li> <li> <p>Specified in the Unix time format.</p> <p>For example,
   * if you wish to use an end time of October 1, 2018, at 8 PM UTC, then you input
   * <code>1538424000</code> as the end time.</p> </li> </ul>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  GetRelationalDatabaseLogEventsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Parameter to specify if the log should start from head or tail. If
   * <code>true</code> is specified, the log event starts from the head of the log.
   * If <code>false</code> is specified, the log event starts from the tail of the
   * log.</p>  <p>For PostgreSQL, the default value of <code>false</code> is
   * the only option available.</p>
   */
  inline bool GetStartFromHead() const { return m_startFromHead; }
  inline bool StartFromHeadHasBeenSet() const { return m_startFromHeadHasBeenSet; }
  inline void SetStartFromHead(bool value) {
    m_startFromHeadHasBeenSet = true;
    m_startFromHead = value;
  }
  inline GetRelationalDatabaseLogEventsRequest& WithStartFromHead(bool value) {
    SetStartFromHead(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to advance to the next or previous page of results from your
   * request.</p> <p>To get a page token, perform an initial
   * <code>GetRelationalDatabaseLogEvents</code> request. If your results are
   * paginated, the response will return a next forward token and/or next backward
   * token that you can specify as the page token in a subsequent request.</p>
   */
  inline const Aws::String& GetPageToken() const { return m_pageToken; }
  inline bool PageTokenHasBeenSet() const { return m_pageTokenHasBeenSet; }
  template <typename PageTokenT = Aws::String>
  void SetPageToken(PageTokenT&& value) {
    m_pageTokenHasBeenSet = true;
    m_pageToken = std::forward<PageTokenT>(value);
  }
  template <typename PageTokenT = Aws::String>
  GetRelationalDatabaseLogEventsRequest& WithPageToken(PageTokenT&& value) {
    SetPageToken(std::forward<PageTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_relationalDatabaseName;

  Aws::String m_logStreamName;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  bool m_startFromHead{false};

  Aws::String m_pageToken;
  bool m_relationalDatabaseNameHasBeenSet = false;
  bool m_logStreamNameHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_startFromHeadHasBeenSet = false;
  bool m_pageTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
