﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/evidently/CloudWatchEvidentlyRequest.h>
#include <aws/evidently/CloudWatchEvidently_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvidently {
namespace Model {

/**
 */
class GetLaunchRequest : public CloudWatchEvidentlyRequest {
 public:
  AWS_CLOUDWATCHEVIDENTLY_API GetLaunchRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetLaunch"; }

  AWS_CLOUDWATCHEVIDENTLY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the launch that you want to see the details of.</p>
   */
  inline const Aws::String& GetLaunch() const { return m_launch; }
  inline bool LaunchHasBeenSet() const { return m_launchHasBeenSet; }
  template <typename LaunchT = Aws::String>
  void SetLaunch(LaunchT&& value) {
    m_launchHasBeenSet = true;
    m_launch = std::forward<LaunchT>(value);
  }
  template <typename LaunchT = Aws::String>
  GetLaunchRequest& WithLaunch(LaunchT&& value) {
    SetLaunch(std::forward<LaunchT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the project that contains the launch.</p>
   */
  inline const Aws::String& GetProject() const { return m_project; }
  inline bool ProjectHasBeenSet() const { return m_projectHasBeenSet; }
  template <typename ProjectT = Aws::String>
  void SetProject(ProjectT&& value) {
    m_projectHasBeenSet = true;
    m_project = std::forward<ProjectT>(value);
  }
  template <typename ProjectT = Aws::String>
  GetLaunchRequest& WithProject(ProjectT&& value) {
    SetProject(std::forward<ProjectT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_launch;

  Aws::String m_project;
  bool m_launchHasBeenSet = false;
  bool m_projectHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvidently
}  // namespace Aws
