﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock-agent/BedrockAgentRequest.h>
#include <aws/bedrock-agent/BedrockAgent_EXPORTS.h>
#include <aws/bedrock-agent/model/APISchema.h>
#include <aws/bedrock-agent/model/ActionGroupExecutor.h>
#include <aws/bedrock-agent/model/ActionGroupSignature.h>
#include <aws/bedrock-agent/model/ActionGroupState.h>
#include <aws/bedrock-agent/model/FunctionSchema.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace BedrockAgent {
namespace Model {

/**
 */
class UpdateAgentActionGroupRequest : public BedrockAgentRequest {
 public:
  AWS_BEDROCKAGENT_API UpdateAgentActionGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateAgentActionGroup"; }

  AWS_BEDROCKAGENT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier of the agent for which to update the action group.</p>
   */
  inline const Aws::String& GetAgentId() const { return m_agentId; }
  inline bool AgentIdHasBeenSet() const { return m_agentIdHasBeenSet; }
  template <typename AgentIdT = Aws::String>
  void SetAgentId(AgentIdT&& value) {
    m_agentIdHasBeenSet = true;
    m_agentId = std::forward<AgentIdT>(value);
  }
  template <typename AgentIdT = Aws::String>
  UpdateAgentActionGroupRequest& WithAgentId(AgentIdT&& value) {
    SetAgentId(std::forward<AgentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the agent version for which to update the action
   * group.</p>
   */
  inline const Aws::String& GetAgentVersion() const { return m_agentVersion; }
  inline bool AgentVersionHasBeenSet() const { return m_agentVersionHasBeenSet; }
  template <typename AgentVersionT = Aws::String>
  void SetAgentVersion(AgentVersionT&& value) {
    m_agentVersionHasBeenSet = true;
    m_agentVersion = std::forward<AgentVersionT>(value);
  }
  template <typename AgentVersionT = Aws::String>
  UpdateAgentActionGroupRequest& WithAgentVersion(AgentVersionT&& value) {
    SetAgentVersion(std::forward<AgentVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the action group.</p>
   */
  inline const Aws::String& GetActionGroupId() const { return m_actionGroupId; }
  inline bool ActionGroupIdHasBeenSet() const { return m_actionGroupIdHasBeenSet; }
  template <typename ActionGroupIdT = Aws::String>
  void SetActionGroupId(ActionGroupIdT&& value) {
    m_actionGroupIdHasBeenSet = true;
    m_actionGroupId = std::forward<ActionGroupIdT>(value);
  }
  template <typename ActionGroupIdT = Aws::String>
  UpdateAgentActionGroupRequest& WithActionGroupId(ActionGroupIdT&& value) {
    SetActionGroupId(std::forward<ActionGroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a new name for the action group.</p>
   */
  inline const Aws::String& GetActionGroupName() const { return m_actionGroupName; }
  inline bool ActionGroupNameHasBeenSet() const { return m_actionGroupNameHasBeenSet; }
  template <typename ActionGroupNameT = Aws::String>
  void SetActionGroupName(ActionGroupNameT&& value) {
    m_actionGroupNameHasBeenSet = true;
    m_actionGroupName = std::forward<ActionGroupNameT>(value);
  }
  template <typename ActionGroupNameT = Aws::String>
  UpdateAgentActionGroupRequest& WithActionGroupName(ActionGroupNameT&& value) {
    SetActionGroupName(std::forward<ActionGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a new name for the action group.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateAgentActionGroupRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update the built-in or computer use action for this action group. If you
   * specify a value, you must leave the <code>description</code>,
   * <code>apiSchema</code>, and <code>actionGroupExecutor</code> fields empty for
   * this action group. </p> <ul> <li> <p>To allow your agent to request the user for
   * additional information when trying to complete a task, set this field to
   * <code>AMAZON.UserInput</code>. </p> </li> <li> <p>To allow your agent to
   * generate, run, and troubleshoot code when trying to complete a task, set this
   * field to <code>AMAZON.CodeInterpreter</code>.</p> </li> <li> <p>To allow your
   * agent to use an Anthropic computer use tool, specify one of the following
   * values. </p>  <p> Computer use is a new Anthropic Claude model
   * capability (in beta) available with Anthropic Claude 3.7 Sonnet and Claude 3.5
   * Sonnet v2 only. When operating computer use functionality, we recommend taking
   * additional security precautions, such as executing computer actions in virtual
   * environments with restricted data access and limited internet connectivity. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-computer-use.html">Configure
   * an Amazon Bedrock Agent to complete tasks with computer use tools</a>. </p>
   *  <ul> <li> <p> <code>ANTHROPIC.Computer</code> - Gives the agent
   * permission to use the mouse and keyboard and take screenshots.</p> </li> <li>
   * <p> <code>ANTHROPIC.TextEditor</code> - Gives the agent permission to view,
   * create and edit files.</p> </li> <li> <p> <code>ANTHROPIC.Bash</code> - Gives
   * the agent permission to run commands in a bash shell.</p> </li> </ul> </li>
   * </ul> <p>During orchestration, if your agent determines that it needs to invoke
   * an API in an action group, but doesn't have enough information to complete the
   * API request, it will invoke this action group instead and return an <a
   * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html">Observation</a>
   * reprompting the user for more information.</p>
   */
  inline ActionGroupSignature GetParentActionGroupSignature() const { return m_parentActionGroupSignature; }
  inline bool ParentActionGroupSignatureHasBeenSet() const { return m_parentActionGroupSignatureHasBeenSet; }
  inline void SetParentActionGroupSignature(ActionGroupSignature value) {
    m_parentActionGroupSignatureHasBeenSet = true;
    m_parentActionGroupSignature = value;
  }
  inline UpdateAgentActionGroupRequest& WithParentActionGroupSignature(ActionGroupSignature value) {
    SetParentActionGroupSignature(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration settings for a computer use action.</p>  <p>
   * Computer use is a new Anthropic Claude model capability (in beta) available with
   * Claude 3.7 Sonnet and Claude 3.5 Sonnet v2 only. For more information, see <a
   * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-computer-use.html">Configure
   * an Amazon Bedrock Agent to complete tasks with computer use tools</a>. </p>
   *
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetParentActionGroupSignatureParams() const {
    return m_parentActionGroupSignatureParams;
  }
  inline bool ParentActionGroupSignatureParamsHasBeenSet() const { return m_parentActionGroupSignatureParamsHasBeenSet; }
  template <typename ParentActionGroupSignatureParamsT = Aws::Map<Aws::String, Aws::String>>
  void SetParentActionGroupSignatureParams(ParentActionGroupSignatureParamsT&& value) {
    m_parentActionGroupSignatureParamsHasBeenSet = true;
    m_parentActionGroupSignatureParams = std::forward<ParentActionGroupSignatureParamsT>(value);
  }
  template <typename ParentActionGroupSignatureParamsT = Aws::Map<Aws::String, Aws::String>>
  UpdateAgentActionGroupRequest& WithParentActionGroupSignatureParams(ParentActionGroupSignatureParamsT&& value) {
    SetParentActionGroupSignatureParams(std::forward<ParentActionGroupSignatureParamsT>(value));
    return *this;
  }
  template <typename ParentActionGroupSignatureParamsKeyT = Aws::String, typename ParentActionGroupSignatureParamsValueT = Aws::String>
  UpdateAgentActionGroupRequest& AddParentActionGroupSignatureParams(ParentActionGroupSignatureParamsKeyT&& key,
                                                                     ParentActionGroupSignatureParamsValueT&& value) {
    m_parentActionGroupSignatureParamsHasBeenSet = true;
    m_parentActionGroupSignatureParams.emplace(std::forward<ParentActionGroupSignatureParamsKeyT>(key),
                                               std::forward<ParentActionGroupSignatureParamsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Lambda function containing the business
   * logic that is carried out upon invoking the action.</p>
   */
  inline const ActionGroupExecutor& GetActionGroupExecutor() const { return m_actionGroupExecutor; }
  inline bool ActionGroupExecutorHasBeenSet() const { return m_actionGroupExecutorHasBeenSet; }
  template <typename ActionGroupExecutorT = ActionGroupExecutor>
  void SetActionGroupExecutor(ActionGroupExecutorT&& value) {
    m_actionGroupExecutorHasBeenSet = true;
    m_actionGroupExecutor = std::forward<ActionGroupExecutorT>(value);
  }
  template <typename ActionGroupExecutorT = ActionGroupExecutor>
  UpdateAgentActionGroupRequest& WithActionGroupExecutor(ActionGroupExecutorT&& value) {
    SetActionGroupExecutor(std::forward<ActionGroupExecutorT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the action group is available for the agent to invoke or
   * not when sending an <a
   * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html">InvokeAgent</a>
   * request.</p>
   */
  inline ActionGroupState GetActionGroupState() const { return m_actionGroupState; }
  inline bool ActionGroupStateHasBeenSet() const { return m_actionGroupStateHasBeenSet; }
  inline void SetActionGroupState(ActionGroupState value) {
    m_actionGroupStateHasBeenSet = true;
    m_actionGroupState = value;
  }
  inline UpdateAgentActionGroupRequest& WithActionGroupState(ActionGroupState value) {
    SetActionGroupState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains either details about the S3 object containing the OpenAPI schema for
   * the action group or the JSON or YAML-formatted payload defining the schema. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html">Action
   * group OpenAPI schemas</a>.</p>
   */
  inline const APISchema& GetApiSchema() const { return m_apiSchema; }
  inline bool ApiSchemaHasBeenSet() const { return m_apiSchemaHasBeenSet; }
  template <typename ApiSchemaT = APISchema>
  void SetApiSchema(ApiSchemaT&& value) {
    m_apiSchemaHasBeenSet = true;
    m_apiSchema = std::forward<ApiSchemaT>(value);
  }
  template <typename ApiSchemaT = APISchema>
  UpdateAgentActionGroupRequest& WithApiSchema(ApiSchemaT&& value) {
    SetApiSchema(std::forward<ApiSchemaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains details about the function schema for the action group or the JSON
   * or YAML-formatted payload defining the schema.</p>
   */
  inline const FunctionSchema& GetFunctionSchema() const { return m_functionSchema; }
  inline bool FunctionSchemaHasBeenSet() const { return m_functionSchemaHasBeenSet; }
  template <typename FunctionSchemaT = FunctionSchema>
  void SetFunctionSchema(FunctionSchemaT&& value) {
    m_functionSchemaHasBeenSet = true;
    m_functionSchema = std::forward<FunctionSchemaT>(value);
  }
  template <typename FunctionSchemaT = FunctionSchema>
  UpdateAgentActionGroupRequest& WithFunctionSchema(FunctionSchemaT&& value) {
    SetFunctionSchema(std::forward<FunctionSchemaT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_agentId;

  Aws::String m_agentVersion;

  Aws::String m_actionGroupId;

  Aws::String m_actionGroupName;

  Aws::String m_description;

  ActionGroupSignature m_parentActionGroupSignature{ActionGroupSignature::NOT_SET};

  Aws::Map<Aws::String, Aws::String> m_parentActionGroupSignatureParams;

  ActionGroupExecutor m_actionGroupExecutor;

  ActionGroupState m_actionGroupState{ActionGroupState::NOT_SET};

  APISchema m_apiSchema;

  FunctionSchema m_functionSchema;
  bool m_agentIdHasBeenSet = false;
  bool m_agentVersionHasBeenSet = false;
  bool m_actionGroupIdHasBeenSet = false;
  bool m_actionGroupNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_parentActionGroupSignatureHasBeenSet = false;
  bool m_parentActionGroupSignatureParamsHasBeenSet = false;
  bool m_actionGroupExecutorHasBeenSet = false;
  bool m_actionGroupStateHasBeenSet = false;
  bool m_apiSchemaHasBeenSet = false;
  bool m_functionSchemaHasBeenSet = false;
};

}  // namespace Model
}  // namespace BedrockAgent
}  // namespace Aws
