﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProviderRequest.h>
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CognitoIdentityProvider {
namespace Model {

/**
 */
class AdminSetUserPasswordRequest : public CognitoIdentityProviderRequest {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API AdminSetUserPasswordRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AdminSetUserPassword"; }

  AWS_COGNITOIDENTITYPROVIDER_API Aws::String SerializePayload() const override;

  AWS_COGNITOIDENTITYPROVIDER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the user pool where you want to set the user's password.</p>
   */
  inline const Aws::String& GetUserPoolId() const { return m_userPoolId; }
  inline bool UserPoolIdHasBeenSet() const { return m_userPoolIdHasBeenSet; }
  template <typename UserPoolIdT = Aws::String>
  void SetUserPoolId(UserPoolIdT&& value) {
    m_userPoolIdHasBeenSet = true;
    m_userPoolId = std::forward<UserPoolIdT>(value);
  }
  template <typename UserPoolIdT = Aws::String>
  AdminSetUserPasswordRequest& WithUserPoolId(UserPoolIdT&& value) {
    SetUserPoolId(std::forward<UserPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the user that you want to query or modify. The value of this
   * parameter is typically your user's username, but it can be any of their alias
   * attributes. If <code>username</code> isn't an alias attribute in your user pool,
   * this value must be the <code>sub</code> of a local user or the username of a
   * user from a third-party IdP.</p>
   */
  inline const Aws::String& GetUsername() const { return m_username; }
  inline bool UsernameHasBeenSet() const { return m_usernameHasBeenSet; }
  template <typename UsernameT = Aws::String>
  void SetUsername(UsernameT&& value) {
    m_usernameHasBeenSet = true;
    m_username = std::forward<UsernameT>(value);
  }
  template <typename UsernameT = Aws::String>
  AdminSetUserPasswordRequest& WithUsername(UsernameT&& value) {
    SetUsername(std::forward<UsernameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new temporary or permanent password that you want to set for the user.
   * You can't remove the password for a user who already has a password so that they
   * can only sign in with passwordless methods. In this scenario, you must create a
   * new user without a password.</p>
   */
  inline const Aws::String& GetPassword() const { return m_password; }
  inline bool PasswordHasBeenSet() const { return m_passwordHasBeenSet; }
  template <typename PasswordT = Aws::String>
  void SetPassword(PasswordT&& value) {
    m_passwordHasBeenSet = true;
    m_password = std::forward<PasswordT>(value);
  }
  template <typename PasswordT = Aws::String>
  AdminSetUserPasswordRequest& WithPassword(PasswordT&& value) {
    SetPassword(std::forward<PasswordT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to <code>true</code> to set a password that the user can immediately sign
   * in with. Set to <code>false</code> to set a temporary password that the user
   * must change on their next sign-in.</p>
   */
  inline bool GetPermanent() const { return m_permanent; }
  inline bool PermanentHasBeenSet() const { return m_permanentHasBeenSet; }
  inline void SetPermanent(bool value) {
    m_permanentHasBeenSet = true;
    m_permanent = value;
  }
  inline AdminSetUserPasswordRequest& WithPermanent(bool value) {
    SetPermanent(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_userPoolId;
  bool m_userPoolIdHasBeenSet = false;

  Aws::String m_username;
  bool m_usernameHasBeenSet = false;

  Aws::String m_password;
  bool m_passwordHasBeenSet = false;

  bool m_permanent{false};
  bool m_permanentHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
