package client

import (
	"context"
	"io"
	"net/url"
	"time"

	"github.com/moby/moby/client/internal/timestamp"
)

// TaskLogsOptions holds parameters to filter logs with.
type TaskLogsOptions struct {
	ShowStdout bool
	ShowStderr bool
	Since      string
	Until      string
	Timestamps bool
	Follow     bool
	Tail       string
	Details    bool
}

// TaskLogsResult holds the result of a task logs operation.
// It implements [io.ReadCloser].
type TaskLogsResult interface {
	io.ReadCloser
}

// TaskLogs returns the logs generated by a service in a [TaskLogsResult].
// as an [io.ReadCloser]. Callers should close the stream.
//
// The underlying [io.ReadCloser] is automatically closed if the context is canceled,
func (cli *Client) TaskLogs(ctx context.Context, taskID string, options TaskLogsOptions) (TaskLogsResult, error) {
	// TODO(thaJeztah): this function needs documentation about the format of ths stream (similar to for container logs)
	// TODO(thaJeztah): migrate CLI utilities to the client where suitable; https://github.com/docker/cli/blob/v29.0.0-rc.1/cli/command/service/logs.go#L73-L348

	query := url.Values{}
	if options.ShowStdout {
		query.Set("stdout", "1")
	}

	if options.ShowStderr {
		query.Set("stderr", "1")
	}

	if options.Since != "" {
		ts, err := timestamp.GetTimestamp(options.Since, time.Now())
		if err != nil {
			return nil, err
		}
		query.Set("since", ts)
	}

	if options.Timestamps {
		query.Set("timestamps", "1")
	}

	if options.Details {
		query.Set("details", "1")
	}

	if options.Follow {
		query.Set("follow", "1")
	}
	query.Set("tail", options.Tail)

	resp, err := cli.get(ctx, "/tasks/"+taskID+"/logs", query, nil)
	if err != nil {
		return nil, err
	}
	return &taskLogsResult{
		ReadCloser: newCancelReadCloser(ctx, resp.Body),
	}, nil
}

type taskLogsResult struct {
	io.ReadCloser
}

var (
	_ io.ReadCloser       = (*taskLogsResult)(nil)
	_ ContainerLogsResult = (*taskLogsResult)(nil)
)
